#include "pch.h"
#include "AreasManager.h"
#include "LinksManager.h"
#include "AreasSAX.h"

/**********************
  CAreasContentHandler
 **********************/


CAreasContentHandler::CAreasContentHandler(CMiniSAXParser* aParser) {
	parser      = aParser;
	state       = ROOT_STATE;
	subHandler  = NULL;
}


CAreasContentHandler::~CAreasContentHandler() {
	if (subHandler != NULL) {
		delete subHandler;
	}
}


void CAreasContentHandler::startElement(const char* name, const CAttributeList& attributes)
        throw (CMiniSAXException)
{
	if (subHandler != NULL) {
		subHandler->startElement( name, attributes );
	}
	else {
		switch (state) {
		case ROOT_STATE:
			if (strcmp( name, "areas" ) == 0) {
				state = AREAS_STATE;
				return;
			}
			break;
		case AREAS_STATE:
			if (strcmp( name, "group" ) == 0) {
				subHandler = new CGroupContentHandler( parser );
				subHandler->startElement( name, attributes );
				return;
			}
		}
		// if we are here then we have unexpected element
		throw CMiniSAXException( "Unexpected element", parser->getCurrentColumn(), parser->getCurrentRow() );
	}
}


void CAreasContentHandler::endElement(const char* name)
        throw (CMiniSAXException)
{
	if (subHandler != NULL) {
		subHandler->endElement(name);
		if (subHandler->elementEnded()) {
			delete subHandler;
			subHandler = NULL;
		}
	}
	else {
		switch (state) {
		case AREAS_STATE:
			if (strcmp( name, "areas" ) == 0) {
				state = ROOT_STATE;
				return;
			}
			break;
		}
		// if we are here then we have unexpected element
		throw CMiniSAXException( "Unexpected end of element", parser->getCurrentColumn(), parser->getCurrentRow() );
	}
}


void CAreasContentHandler::handleCharacters(const char* buffer, size_t len)
        throw (CMiniSAXException)
{
	if (subHandler != NULL) {
		subHandler->handleCharacters( buffer, len );
	}
	else {
	}
}


/****************************
  CGroupContentHandler
 ****************************/


CGroupContentHandler::CGroupContentHandler(CMiniSAXParser* aParser) : CSubContentHandler( aParser )
{
	state = ROOT_STATE;
	group = NULL;
}


CGroupContentHandler::~CGroupContentHandler() {
}


void CGroupContentHandler::startElement(const char* name, const CAttributeList& attributes)
    throw (CMiniSAXException)
{
	if (subHandler != NULL) {
		subHandler->startElement( name, attributes );
	} else {
		switch (state) {
		case ROOT_STATE:
			if (strcmp( name, "group" ) == 0) {
				// name (required)
				const CAttribute*	nameAttr = attributes.getAttribute( "name" );
				if (nameAttr == NULL) {
					throw CMiniSAXException( "Group name expected (as attribute)", 
											 parser->getCurrentColumn(), parser->getCurrentRow() );
				}
				if (CAreasManager::getInstance()->getGroup( nameAttr->getValue() ) != NULL) {
					throw CMiniSAXException( "Found duplicate group", parser->getCurrentColumn(), parser->getCurrentRow() );
				}
				// TODO: check name length

				// description (optional)
				const CAttribute*	descAttr = attributes.getAttribute( "description" );
				
				group = new CAreaGroup();
				group->setName( nameAttr->getValue() );
				if (descAttr != NULL) {
					group->setDescription( descAttr->getValue() );
				}

				CAreasManager::getInstance()->addGroup( group );

				state = GROUP_STATE;
				return;
			}
			break;
		case GROUP_STATE:
			if (strcmp( name, "area" ) == 0) {
				subHandler = new CAreaContentHandler( parser );
				subHandler->startElement( name, attributes );
				return;
			}
			break;
		}
		// if we are here then we have unexpected element
		throw CMiniSAXException( "Unexpected element", parser->getCurrentColumn(), parser->getCurrentRow() );
	}
}


void CGroupContentHandler::endElement(const char* name)
    throw (CMiniSAXException)
{
	if (subHandler != NULL) {
		subHandler->endElement( name );
		if (subHandler->elementEnded()) {
			CArea*	area = ((CAreaContentHandler*)subHandler)->detachArea();

			delete subHandler;
			subHandler = NULL;

			area->setGroup( group );

			CAreasManager::getInstance()->addArea( area );
		}
	} else {		
		if (state == GROUP_STATE && strcmp( name, "group" ) == 0) {
			group		 = NULL;
			state        = ROOT_STATE;
			endOfElement = true;
			return;
		}
		// if we are here then we have unexpected element
		throw CMiniSAXException( "Unexpected end of element", parser->getCurrentColumn(), parser->getCurrentRow() );
	}
}


void CGroupContentHandler::handleCharacters(const char* buffer, size_t len)
    throw (CMiniSAXException)
{
	if (subHandler != NULL) {
		subHandler->handleCharacters( buffer, len );
	} else {
	}
}


/****************************
  CAreaContentHandler
 ****************************/

CAreaContentHandler::CAreaContentHandler(CMiniSAXParser* aParser) : CSubContentHandler( aParser )
{
	state = ROOT_STATE;
	area  = NULL;
}


CAreaContentHandler::~CAreaContentHandler() {
	if (area != NULL) {
		delete area;
	}
}


void CAreaContentHandler::startElement(const char* name, const CAttributeList& attributes)
    throw (CMiniSAXException)
{
	switch (state) {
	case ROOT_STATE:
		if (strcmp( name, "area" ) == 0) {
			// name (required)
			const CAttribute*	nameAttr = attributes.getAttribute( "name" );
			if (nameAttr == NULL) {
				throw CMiniSAXException( "Area name expected (as attribute)", 
										 parser->getCurrentColumn(), parser->getCurrentRow() );
			}
			// TODO: check name length, check name integrity (no spaces!)

			// description (optional)
			const CAttribute*	descAttr = attributes.getAttribute( "description" );
			
			// path (required)
			const CAttribute*	pathAttr = attributes.getAttribute( "path" );
			if (pathAttr == NULL) {
				throw CMiniSAXException( "Path expected (as attribute)", 
										 parser->getCurrentColumn(), parser->getCurrentRow() );
			}
			// send priority (optional)
			int	sendPriority = 0;
			const CAttribute*	sendPriorityAttr = attributes.getAttribute( "sendPriority" );
			if (sendPriorityAttr != NULL) {
				int	scanResult = sscanf( sendPriorityAttr->getValue(), "%d", &sendPriority );
				if (scanResult == 0 || scanResult == EOF || sendPriority < 0) {
					throw CMiniSAXException( "Invalid send priority", 
											 parser->getCurrentColumn(), parser->getCurrentRow() );
				}
			}
			// receive priority (optional)
			int	receivePriority = 0;
			const CAttribute*	receivePriorityAttr = attributes.getAttribute( "receivePriority" );
			if (receivePriorityAttr != NULL) {
				int	scanResult = sscanf( receivePriorityAttr->getValue(), "%d", &receivePriority );
				if (scanResult == 0 || scanResult == EOF || receivePriority < 0) {
					throw CMiniSAXException( "Invalid receive priority", 
											 parser->getCurrentColumn(), parser->getCurrentRow() );
				}
			}

			// create and setup area
			LPTSTR	buffer;

			area = new CArea();
			// name
			buffer = new TCHAR[strlen(nameAttr->getValue()) + 1];
			::OemToChar( nameAttr->getValue(), buffer );
			area->setName( buffer );
			delete [] buffer;
			// description
			if (descAttr != NULL) {
				buffer = new TCHAR[strlen(descAttr->getValue()) + 1];
				::OemToChar( descAttr->getValue(), buffer );
				area->setDescription( buffer );
				delete [] buffer;
			}
			area->setPath( pathAttr->getValue() );
			area->setSendPriority( sendPriority );
			area->setReceivePriority( receivePriority );

			// these properties can be changed by inner elements
			area->resetFlag( CArea::FLAG_PASSTHROUGH       );
			area->resetFlag( CArea::FLAG_VETO_MANUAL_PURGE );
			area->resetFlag( CArea::FLAG_OPEN_FOR_RESEND   );

			state = AREA_STATE;
			return;
		}
		break;
	case AREA_STATE:
		if (strcmp( name, "passthrough" ) == 0) {
			area->setFlag( CArea::FLAG_PASSTHROUGH );
			state = PASSTHROUGH_STATE;
			return;
		}
		if (strcmp( name, "vetoManualPurge" ) == 0) {
			area->setFlag( CArea::FLAG_VETO_MANUAL_PURGE );
			state = VETO_MANUAL_PURGE_STATE;
			return;
		}
		if (strcmp( name, "openForResend" ) == 0) {
			area->setFlag( CArea::FLAG_OPEN_FOR_RESEND );
			state = OPEN_FOR_RESEND_STATE;
			return;
		}
		if (strcmp( name, "outbound" ) == 0) {
			const CAttribute*	typeAttr = attributes.getAttribute( "type" );
			if (typeAttr != NULL) {
				if (strcmp( typeAttr->getValue(), "binkley" ) == 0) {
					area->setOutboundDefined( true );
					area->setOutboundType( OT_BINKLEY );
				} else
				if (strcmp( typeAttr->getValue(), "fileBoxes" ) == 0) {
					area->setOutboundDefined( true );
					area->setOutboundType( OT_FILE_BOXES );
				} else {
					throw CMiniSAXException( "Unknown outbound type", parser->getCurrentColumn(), parser->getCurrentRow() );
				}
			} else {
				throw CMiniSAXException( "type attribute absent", parser->getCurrentColumn(), parser->getCurrentRow() );
			}

			state = OUTBOUND_STATE;
			return;
		}
		if (strcmp( name, "linked" ) == 0) {
			state = LINKED_STATE;

			// remember routing for link if specified

			const CAttribute*	routingAttr = attributes.getAttribute( "routing" );
			if (routingAttr != NULL) {
				if (strcmp( routingAttr->getValue(), "send" ) == 0) {
					linkRouting = ROUTE_SEND;
				} else
				if (strcmp( routingAttr->getValue(), "receive" ) == 0) {
					linkRouting = ROUTE_RECEIVE;
				} else
				if (strcmp( routingAttr->getValue(), "bidirectional" ) == 0) {
					linkRouting = ROUTE_BIDIRECT;
				} else {
					throw CMiniSAXException( "Unknown routing", parser->getCurrentColumn(), parser->getCurrentRow() );
				}				
			} else {
				// individual routing not specified
				linkRouting = ROUTE_NONE;
			}
			 
			return;
		}
		break;
	}
	// if we are here then we have unexpected element
	throw CMiniSAXException( "Unexpected element", parser->getCurrentColumn(), parser->getCurrentRow() );
}


void CAreaContentHandler::endElement(const char* name)
    throw (CMiniSAXException)
{
	switch (state) {
	case LINKED_STATE:
		if (strcmp( name, "linked" ) == 0) {
			state = AREA_STATE;
			return;
		}
		break;
	case OUTBOUND_STATE:
		if (strcmp( name, "outbound" ) == 0) {
			state = AREA_STATE;
			return;
		}
		break;
	case OPEN_FOR_RESEND_STATE:
		if (strcmp( name, "openForResend" ) == 0) {
			state = AREA_STATE;
			return;
		}
		break;
	case VETO_MANUAL_PURGE_STATE:
		if (strcmp( name, "vetoManualPurge" ) == 0) {
			state = AREA_STATE;
			return;
		}
		break;
	case PASSTHROUGH_STATE:
		if (strcmp( name, "passthrough" ) == 0) {
			state = AREA_STATE;
			return;
		}
		break;
	case AREA_STATE:
		if (strcmp( name, "area" ) == 0) {
			state = ROOT_STATE;
			endOfElement = true;
			return;
		}
		break;
	}
	// if we are here then we have unexpected element
	throw CMiniSAXException( "Unexpected end of element", parser->getCurrentColumn(), parser->getCurrentRow() );
}


void CAreaContentHandler::handleCharacters(const char* buffer, size_t len)
    throw (CMiniSAXException)
{
	if (state == LINKED_STATE) {
		SFTNAddress	linkAddress;
		if (!CLinksManager::getInstance()->parseLinkAddress( tstring( buffer, len).c_str(), linkAddress )) {
			throw CMiniSAXException( "Invalid link address", parser->getCurrentColumn(), parser->getCurrentRow() );
		}

		PLink	link = CLinksManager::getInstance()->getLink( linkAddress );
		if (link == NULL) {
			throw CMiniSAXException( "Unknown link", parser->getCurrentColumn(), parser->getCurrentRow() );
		}

		TiePtr	tie = new CTie();
		tie->setLink( link );
		tie->setRouteType( linkRouting );

		area->addTie( tie );
	}
}


PArea CAreaContentHandler::detachArea() {
	PArea	areaRef = area;
	area = NULL;
	return areaRef;
}
